/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_PSN_ACCOUNT_INTERFACE_H
#define _SCE_PSN_ACCOUNT_INTERFACE_H

#include "np_toolkit_defines.h"

/// Namespace for the %NP %Toolkit project separate from the utilities
namespace sce {
	namespace Toolkit{
		namespace NP{

			/// @brief
			/// The namespace containing PSN user profile functionality.
			///	
			/// The namespace containing PSN user profile functionality.
			namespace UserProfile {
			/// @brief
			/// The external interface to PSN user profile functionality.
			///
			/// This class contains the set of static functions for accessing
			/// information pertaining to the users PSN account.
			class Interface
			{
			public:

				/// @brief 
				/// Gets the PSN user's Online ID.
				///
				/// Gets the PSN user's Online ID. The Online ID is selected by the user when signing up to
				/// PlayStationNetwork. It is composed of 3 to 16 characters and can be made up of alphanumeric 
				/// characters, hyphens and underscores. An Online ID is guaranteed to be unique.
				/// 
				/// @param	onlineId	An array that must be of size <c>SCE_NET_NP_ONLINEID_MAX_LENGTH</c>.
				/// @param  async		A flag that indicates whether the function will be called synchronously or asynchronously. 
				/// 
				/// @retval				SCE_TOOLKIT_NP_SUCCESS				The operation was successful. 
				/// @retval				SCE_TOOLKIT_NP_OFFLINE				The operation failed because the network was unavailable.
				/// @retval				SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed as an argument.
				/// @retval				SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library was not initialized.
				static int getOnlineId(sce::Toolkit::NP::Utilities::Future<SceNpOnlineId> *onlineId, bool async = true);

				/// @brief 
				/// Gets the PSN user's %NP ID.
				///
				/// Gets the PSN user's %NP ID. The %NP ID appends option fields and version information necessary
				/// for server access to the Online ID. It is used by the system utilities of PlayStationNetwork
				/// for identifying the user.
				/// 
				/// @param npid		A pointer to the <c>SceNpId</c>.
				/// @param async	A flag that indicates whether the function will be called synchronously or asynchronously. 
				///
				/// @retval				SCE_TOOLKIT_NP_SUCCESS				The operation was successful. 
				/// @retval				SCE_TOOLKIT_NP_OFFLINE				The operation failed because the network was unavailable.
				/// @retval				SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed as an argument.
				/// @retval				SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library was not initialized.
				static int getNpId(sce::Toolkit::NP::Utilities::Future<SceNpId> *npid, bool async = true);

				/// @brief 
				/// Gets the PSN user's Avatar URL.
				///
				/// Gets the PSN user's Avatar URL, which is the URL for the user's image. The avatar is used as 
				/// an icon to represent the user's account on the system software.
				/// 
				/// @param avatarUrl	A string containing the PSN user's Avatar URL. It must be of size <c>SCE_NET_NP_AVATAR_URL_MAX_LENGTH</c>.
				/// @param async		A flag that indicates whether the function will be called synchronously or asynchronously. 
				///
				/// @retval				SCE_TOOLKIT_NP_SUCCESS				The operation was successful. 
				/// @retval				SCE_TOOLKIT_NP_OFFLINE				The operation failed because the network was unavailable.
				/// @retval				SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed as an argument.
				/// @retval				SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library was not initialized.
				static int getAvatarUrl(sce::Toolkit::NP::Utilities::Future<SceNpAvatarUrl> *avatarUrl, bool async = true);

				/// @brief 
				/// Gets a PSN user's cached info.
				///
				/// Gets a PSN user's cached info. This is information about the %NP user that is saved to the internal hard disk
				/// drive every time the information is updated. The information is also saved when the user signs up
				/// or signs into the PSN.
				/// 
				/// @param userInfo	Output. Receives the PSN user's cached info (<c>onlineId</c>, <c>npId</c>, <c>onlineName</c> and <c>avatarUrl</c>).
				/// @param async	A flag that indicates whether the function will be called synchronously or asynchronously. 
				///
				/// @retval				SCE_TOOLKIT_NP_SUCCESS				The operation was successful. 
				/// @retval				SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed as an argument.
				/// @retval				SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library was not initialized.
				static int getCachedUserInfo(sce::Toolkit::NP::Utilities::Future<SceNpManagerCacheParam> *userInfo, bool async = true);

				/// @brief 
				/// Gets a PSN user's country details.
				///
				/// Gets a PSN user's country details. 
				/// 
				/// @param info		Output. Receives the PSN user's country info, which includes the country code and language.
				///					Language is a <c>CELL_SYSUTIL_LANG_XXX</c> value defined in the system utilities.
				/// @param async	A flag that indicates whether the function will be called synchronously or asynchronously. 
				///
				/// @retval				SCE_TOOLKIT_NP_SUCCESS				The operation was successful. 
				/// @retval				SCE_TOOLKIT_NP_OFFLINE				The operation failed because the network was unavailable.
				/// @retval				SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed as an argument.
				/// @retval				SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library was not initialized.
				static int getCountryInfo(sce::Toolkit::NP::Utilities::Future<CountryInfo> *info, bool async = true );

				/// @brief 
				/// Gets a PSN user's parental control details.
				///
				/// Gets a PSN user's parental control details. This includes content and chat restrictions as well as the user's age.
				/// 
				/// @param info			Output. Receives the PSN user's parental control info.
				/// @param async		A flag that indicates whether the function will be called synchronously or asynchronously. 
				///
				/// @retval				SCE_TOOLKIT_NP_SUCCESS				The operation was successful. 
				/// @retval				SCE_TOOLKIT_NP_OFFLINE				The operation failed because the network was unavailable.
				/// @retval				SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed as an argument.
				/// @retval				SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library was not initialized.
				static int getParentalControlInfo(sce::Toolkit::NP::Utilities::Future<ParentalControlInfo> *info, bool async = true);

				/// @brief 
				/// Gets the current platform the application is running on.
				///
				/// Gets the current platform the application is running on.
				/// 
				/// @param platform		Output. Receives the current platform.
				/// @param async		A flag that indicates whether the function will be called synchronously or asynchronously. Defaults to true.
				///
				/// @retval				SCE_TOOLKIT_NP_SUCCESS				The operation was successful. 
				/// @retval				SCE_TOOLKIT_NP_OFFLINE				The operation failed because the network was unavailable.
				/// @retval				SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed as an argument.
				/// @retval				SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library was not initialized.
				static int getPlatform(sce::Toolkit::NP::Utilities::Future<SceNpPlatformType> *platform, bool async = true);
			};

			}//end UserProfile
		}// end NP
	}// end Toolkit
}// end sce

#endif
